/**
FH3D02_GettingStarted V.1.0 - Library developed for the use
with HallinOne® magnetic field sensors of type FH3D02.

Copyright (C) 2024 Fraunhofer Institute for Integrated Circuits IIS

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

For purpose of clarity, LICENSE.txt only grants you a copyright 
license to the sketches of Fraunhofer. For the use of the 
HallinOne® 3D magnetic field sensors of Fraunhofer, you are required 
to execute another agreement with Fraunhofer. In no event Fraunhofer 
grants you a license in the HallinOne® 3D magnetic field sensors 
according to the LGPL-v2.1 or any other Open Source Software License.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**/

#include <SPI.h>

/* Pin Definitions */
#define PIN_SENSOR_RESET_N   8
#define PIN_SENSOR_READY     9
#define PIN_SENSOR_CS_N     10
#define PIN_SENSOR_MOSI     11
#define PIN_SENSOR_MISO     12
#define PIN_SENSOR_SCK      13

#define MAX_READY_CHECKS    500U

#define FH3D02_CHIP_ID      0x0503U

/* Helper-Macros for Register Access */
#define write_access(reg) ((reg) | 0x8000U)
#define read_access(reg)  ((reg) & 0x7FFFU)

// --------------------------------------------------------------------------------
// Hallinone Sensor
// --------------------------------------------------------------------------------
/* Transmit data via SPI. Received data overwrites data in initial array. */
static void spi_transmit(uint16_t * data, uint16_t data_len)
{
  if (NULL == data || 0 == data_len)
    return;

  digitalWrite(PIN_SENSOR_CS_N, LOW);

  for (uint16_t i = 0; i < data_len; i++) {
    data[i] = SPI.transfer16(data[i]);
  }

  digitalWrite(PIN_SENSOR_CS_N, HIGH);
}


// --------------------------------------------------------------------------------
/* Write 'data' in a sensor register defined by its address 'addr'. */
uint16_t hallinone_SensorWriteRegister(uint16_t addr, uint16_t data)
{
  uint16_t spi_data[2] = {write_access(addr), data};

  spi_transmit(spi_data, 2);
  return spi_data[1];
}


// --------------------------------------------------------------------------------
/* Read a sensor register defined by its address 'addr'. */
uint16_t hallinone_SensorReadRegister(uint16_t addr)
{
  uint16_t spi_data[2] = {read_access(addr), 0x0000};

  spi_transmit(spi_data, 2);
  return spi_data[1];
}


// --------------------------------------------------------------------------------
/* Wait for the sensor until the measurement is finished. */
void hallinone_SensorWaitForReady(void) {
  uint16_t ready_timeout_cnt = 0;

  digitalWrite(PIN_SENSOR_CS_N, LOW);

  while (LOW == digitalRead(PIN_SENSOR_READY)) {
    ready_timeout_cnt++;
    delayMicroseconds(10);

    if (ready_timeout_cnt == MAX_READY_CHECKS) {
      if (Serial) {
        Serial.print("Wait for ready timeout occurred: ");
        Serial.print(MAX_READY_CHECKS * 10, DEC);
        Serial.print("us passed, exiting...\n");
      }
      break;
    }
  }

  digitalWrite(PIN_SENSOR_CS_N, HIGH);
}


// --------------------------------------------------------------------------------
/* Reset sensor. */
void hallinone_SensorReset(void)
{
  digitalWrite(PIN_SENSOR_RESET_N, LOW);
  delay(30);
  digitalWrite(PIN_SENSOR_RESET_N, HIGH);
  delay(10);
}


// --------------------------------------------------------------------------------
/* Check sensor connection (for getting started with breakout board and free lead wiring ...)*/
bool hallinone_CheckSensorConnection(void)
{
  // assumes sensor reset performed just before
  // assumes serial connection etablished
  bool wiring_status;
  bool pin_level;
  uint16_t chip_id, prev_data;
  char msg[128];

  wiring_status = false;

  Serial.print("\nChecking breakout board connection\n");

  // check READY HIGH, attention chip has to be selected
  Serial.print("  Checking READY HIGH after reset ...");
  digitalWrite(PIN_SENSOR_CS_N, LOW);
  delay(10);
  pin_level = digitalRead(PIN_SENSOR_READY);
  digitalWrite(PIN_SENSOR_CS_N, HIGH);
  delay(10);

  if(pin_level != HIGH) {
    Serial.print("\n    READY not HIGH after reset - please check READY and CLK wiring! Stopping ...\n");
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }


  // Check SPI by reading ChipID
  Serial.print("  Checking SPI read ...");
  chip_id = hallinone_SensorWriteRegister(0x0000, 0x0000);

  if(chip_id != FH3D02_CHIP_ID) {
    sprintf(msg, "\n    CHIP_ID is not 0x%04X like expected but 0x%04X. Please check SCK and MISO wiring! Stopping ...\n", FH3D02_CHIP_ID, chip_id);
    Serial.print(msg);
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }


  // check READY LOW after first SPI transfer, attention RM<1:0> = 0 after reset, so chip has to be selected
  Serial.print("  Checking READY LOW after first SPI transfer ...");
  digitalWrite(PIN_SENSOR_CS_N, LOW);
  delay(10);
  pin_level = digitalRead(PIN_SENSOR_READY);
  digitalWrite(PIN_SENSOR_CS_N, HIGH);
  delay(10);

  if(pin_level != LOW) {
    Serial.print("\n    READY not LOW after first SPI transfer - please check READY and CLK wiring! Stopping ...\n");
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }


  // Check SPI by writing data and reading back
  Serial.print("  Checking SPI by writing data and reading back\n");

  Serial.print("    Reading and checking initial content ...");
  prev_data = hallinone_SensorWriteRegister(0x000B, 0x0000); // Rdy2hZ = 0;
  if(prev_data != 0x0002U) {
    sprintf(msg, "\n      Register 0x00B content not 0x0002 after reset but 0x%04X. Please check RESET_N and CLK wiring! Stopping ...\n", prev_data);
    Serial.print(msg);
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }

  Serial.print("    Write data to register 0x00B and read back content ...");
  prev_data = hallinone_SensorReadRegister(0x000B);
  if(prev_data != 0x0000U) {
    sprintf(msg, "\n    Register 0x00B readback not equal to transferred content. Please check MOSI and SCK wiring! Stopping ...\n");
    Serial.print(msg);
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }


  // Re-Check RESET_N
  Serial.print("  Checking reset again by reading register content after reset ...");
  hallinone_SensorReset();
  prev_data = hallinone_SensorReadRegister(0x000B);
  if(prev_data != 0x0002U) {
    sprintf(msg, "\n    Register 0x00B content not 0x0002 after reset but 0x%04X. Please check RESET_N and CLK wiring! Stopping ...\n", prev_data);
    Serial.print(msg);
    return wiring_status;
  }
  else {
    Serial.print(" done!\n");
  }


  // all tests successfully done
  wiring_status = true;
  Serial.print("Breakout board connection checks done. Supply may be checked for example by looking at noise of measurement values\n\n");

  // return status
  return wiring_status;
}


// --------------------------------------------------------------------------------
// Setup & Loop
// --------------------------------------------------------------------------------
/* Initlialize board and sensor.*/
void setup() {
  Serial.begin(115200, SERIAL_8N1); // Baudrate of Serial Interface (possible values: 9600, 19200, 38400, 76800, 115200, 250000)
  Serial.setTimeout(1000);        // Maximum time to block waiting for serial data (in milliseconds)
  while (!Serial) {
    ; // wait for serial port to connect. Needed for native USB
  }

  /* Configure Pins */
  pinMode(PIN_SENSOR_RESET_N, OUTPUT);
  pinMode(PIN_SENSOR_READY, INPUT);
  pinMode(PIN_SENSOR_CS_N, OUTPUT);
  pinMode(PIN_SENSOR_MOSI, OUTPUT);
  pinMode(PIN_SENSOR_MISO, INPUT);
  pinMode(PIN_SENSOR_SCK, OUTPUT);

  /* Set default Levels */
  digitalWrite(PIN_SENSOR_CS_N, HIGH);
  digitalWrite(PIN_SENSOR_RESET_N, HIGH);

  /* Initialize SPI for communication with Sensor */
  SPI.begin();
  SPI.setBitOrder(MSBFIRST);
  SPI.setClockDivider(SPI_CLOCK_DIV16); // F_CPU = 16MHz -> SPI Clock = 1MHz
  SPI.setDataMode(SPI_MODE0);

  // Initialize FH3D12 according to datasheet Chapter Application Notes / Basic configuration
  hallinone_SensorReset();

  // Check breakout board connections
  if(hallinone_CheckSensorConnection() != true) {
    while(1); // stopp due to errors
  }

  /* Headline for Serial plotter */
  Serial.print("T\tZ0\tY0\tX0\tZ1\tY1\tX1\n");

  // Start Sequencer in "Dual Magnetic Probe" mode
  hallinone_SensorWriteRegister(0x000B, 0x0340); // TableSelect = 1; Special = 3;
  hallinone_SensorWriteRegister(0x000E, 0x0002); // ContMeas = 1;
}
// --------------------------------------------------------------------------------

/* Main loop with simple measurement.*/
void loop() {
  uint16_t temp, bz0, by0, bx0, bz1, by1, bx1;

  // Wait for READY
  hallinone_SensorWaitForReady();

  // Temperature
  temp = hallinone_SensorReadRegister(0x0110);
  
  // Pixel 0
  bz0 = hallinone_SensorReadRegister(0x0124);
  by0 = hallinone_SensorReadRegister(0x0112);
  bx0 = hallinone_SensorReadRegister(0x0114);

  // Pixel 1
  by1 = hallinone_SensorReadRegister(0x0111);
  bx1 = hallinone_SensorReadRegister(0x0113);
  bz1 = hallinone_SensorReadRegister(0x0122); // Register 0x122 last for releasing READY

  // Print
  char msg[128];
  sprintf(msg, "%d\t%d\t%d\t%d\t%d\t%d\t%d\n", temp, bz0, by0, bx0, bz1, by1, bx1);
  Serial.print (msg);
}
// --------------------------------------------------------------------------------
